<?php
/**
 * Little Hands Studio Child Theme Functions
 */

// Enqueue parent and child theme styles
function bootscore_child_enqueue_styles() {
    wp_enqueue_style('bootscore-style', get_template_directory_uri() . '/style.css');
    wp_enqueue_style('bootscore-child-style', get_stylesheet_directory_uri() . '/style.css', array('bootscore-style'));

    // Enqueue Google Fonts
    wp_enqueue_style('google-fonts', 'https://fonts.googleapis.com/css2?family=Montserrat:wght@300;400;500;600;700&display=swap', array(), null);

    // Enqueue Font Awesome
    wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css', array(), null);
}
add_action('wp_enqueue_scripts', 'bootscore_child_enqueue_styles');

// Enqueue custom JavaScript
function bootscore_child_enqueue_scripts() {
    wp_enqueue_script('bootscore-child-scripts', get_stylesheet_directory_uri() . '/js/custom.js', array('jquery'), '1.1', true);

    // Enqueue WooCommerce add to cart JS if WooCommerce is active
    if (class_exists('WooCommerce')) {
        wp_enqueue_script('wc-add-to-cart');

        // Make sure this loads on checkout
        if (is_checkout()) {
            wp_enqueue_script('woocommerce');
            wp_enqueue_script('wc-checkout');
        }

        wp_localize_script('bootscore-child-scripts', 'wc_add_to_cart_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'wc_ajax_url' => WC_AJAX::get_endpoint('%%endpoint%%'),
            'cart_url' => wc_get_cart_url(),
            'i18n_view_cart' => esc_attr__('View cart', 'woocommerce'),
            'wc_ajax_nonce' => wp_create_nonce('wc_store_api'),
            'lhs_products_nonce' => wp_create_nonce('lhs_products_nonce')
        ));
    }
}
add_action('wp_enqueue_scripts', 'bootscore_child_enqueue_scripts');

// Add theme support for post thumbnails
add_theme_support('post-thumbnails');

// Register navigation menus
function bootscore_child_register_menus() {
    register_nav_menus(array(
        'primary' => __('Primary Menu', 'bootscore-child'),
        'footer' => __('Footer Menu', 'bootscore-child'),
    ));
}
add_action('init', 'bootscore_child_register_menus');

// Custom image sizes
add_image_size('hero-image', 500, 500, true);
add_image_size('promo-image', 400, 500, true);
add_image_size('workshop-image', 400, 400, true);

// Remove default WordPress block styles if needed
function bootscore_child_remove_block_styles() {
    wp_dequeue_style('wp-block-library');
    wp_dequeue_style('wp-block-library-theme');
}
add_action('wp_enqueue_scripts', 'bootscore_child_remove_block_styles', 100);

// Custom body classes for pages
function bootscore_child_body_classes($classes) {
    if (is_front_page()) {
        $classes[] = 'little-hands-home';
    }
    if (is_checkout()) {
        $classes[] = 'checkout-page';
        $classes[] = 'little-hands-checkout';
    }
    return $classes;
}
add_filter('body_class', 'bootscore_child_body_classes');

// Register widget areas
function bootscore_child_register_widgets() {
    register_sidebar(array(
        'name'          => __('Footer Social Widget', 'bootscore-child'),
        'id'            => 'footer-social-widget',
        'description'   => __('Add social media links and icons here.', 'bootscore-child'),
        'before_widget' => '<div class="social-widget">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ));
}
add_action('widgets_init', 'bootscore_child_register_widgets');

// Create custom social links widget
class Little_Hands_Social_Widget extends WP_Widget {

    public function __construct() {
        parent::__construct(
            'little_hands_social',
            'Little Hands Social Links',
            array('description' => 'Add social media links with Font Awesome icons')
        );
    }

    public function widget($args, $instance) {
        echo $args['before_widget'];

        $social_platforms = array(
            'email' => array('icon' => 'fas fa-envelope', 'prefix' => 'mailto:'),
            'instagram' => array('icon' => 'fab fa-instagram', 'prefix' => ''),
            'facebook' => array('icon' => 'fab fa-facebook', 'prefix' => ''),
            'twitter' => array('icon' => 'fab fa-twitter', 'prefix' => ''),
            'youtube' => array('icon' => 'fab fa-youtube', 'prefix' => ''),
            'tiktok' => array('icon' => 'fab fa-tiktok', 'prefix' => ''),
            'pinterest' => array('icon' => 'fab fa-pinterest', 'prefix' => ''),
            'linkedin' => array('icon' => 'fab fa-linkedin', 'prefix' => '')
        );

        foreach ($social_platforms as $platform => $details) {
            $url = !empty($instance[$platform]) ? $instance[$platform] : '';
            if ($url) {
                $full_url = ($platform === 'email') ? $details['prefix'] . $url : $url;
                $target = ($platform === 'email') ? '' : ' target="_blank"';
                echo '<a href="' . esc_url($full_url) . '" title="' . ucfirst($platform) . '"' . $target . '><i class="' . $details['icon'] . '"></i></a>';
            }
        }

        echo $args['after_widget'];
    }

    public function form($instance) {
        $social_platforms = array(
            'email' => 'Email Address',
            'instagram' => 'Instagram URL',
            'facebook' => 'Facebook URL',
            'twitter' => 'Twitter URL',
            'youtube' => 'YouTube URL',
            'tiktok' => 'TikTok URL',
            'pinterest' => 'Pinterest URL',
            'linkedin' => 'LinkedIn URL'
        );

        foreach ($social_platforms as $platform => $label) {
            $value = !empty($instance[$platform]) ? $instance[$platform] : '';
            echo '<p>';
            echo '<label for="' . $this->get_field_id($platform) . '">' . $label . ':</label>';
            echo '<input class="widefat" id="' . $this->get_field_id($platform) . '" name="' . $this->get_field_name($platform) . '" type="text" value="' . esc_attr($value) . '" />';
            echo '</p>';
        }
    }

    public function update($new_instance, $old_instance) {
        $instance = array();
        $social_platforms = array('email', 'instagram', 'facebook', 'twitter', 'youtube', 'tiktok', 'pinterest', 'linkedin');

        foreach ($social_platforms as $platform) {
            $instance[$platform] = !empty($new_instance[$platform]) ? sanitize_text_field($new_instance[$platform]) : '';
        }

        return $instance;
    }
}

// Register the social widget
function register_little_hands_social_widget() {
    register_widget('Little_Hands_Social_Widget');
}
add_action('widgets_init', 'register_little_hands_social_widget');

// Custom AJAX handler for adding products to cart
function little_hands_add_to_cart() {
    if (!wp_verify_nonce($_POST['nonce'], 'wc_store_api')) {
        wp_die('Security check failed');
    }

    $product_id = intval($_POST['product_id']);
    $variation_id = isset($_POST['variation_id']) ? intval($_POST['variation_id']) : 0;
    $quantity = intval($_POST['quantity']);

    if ($variation_id) {
        // Adding variation
        $result = WC()->cart->add_to_cart($product_id, $quantity, $variation_id);
    } else {
        // Adding simple product
        $result = WC()->cart->add_to_cart($product_id, $quantity);
    }

    if ($result) {
        wp_send_json_success(array(
            'message' => 'Product added to cart',
            'cart_count' => WC()->cart->get_cart_contents_count()
        ));
    } else {
        wp_send_json_error(array('message' => 'Failed to add product to cart'));
    }
}
add_action('wp_ajax_little_hands_add_to_cart', 'little_hands_add_to_cart');
add_action('wp_ajax_nopriv_little_hands_add_to_cart', 'little_hands_add_to_cart');

// Fix Bootscore quantity button warnings
function bootscore_child_fix_qty_args($args) {
    if (!isset($args['input_value'])) {
        $args['input_value'] = 1;
    }
    return $args;
}
add_filter('woocommerce_quantity_input_args', 'bootscore_child_fix_qty_args', 10, 1);

// Customize WooCommerce checkout and cart
function little_hands_checkout_customizations() {
    // Remove sidebar on checkout and cart
    if (is_checkout() || is_cart()) {
        remove_action('woocommerce_sidebar', 'woocommerce_get_sidebar', 10);
        add_filter('bootscore_sidebar', '__return_false');

        // Force full width layout
        add_filter('bootscore_container_class', function($class) {
            return 'container-fluid';
        });

        // Remove breadcrumbs and page title
        remove_action('woocommerce_before_main_content', 'woocommerce_breadcrumb', 20);
        add_filter('woocommerce_show_page_title', '__return_false');
    }
}
add_action('wp', 'little_hands_checkout_customizations');

// Add custom checkout styling
function little_hands_checkout_styles() {
    if (is_checkout()) {
        wp_enqueue_style('little-hands-checkout', get_stylesheet_directory_uri() . '/css/checkout.css', array(), '1.0');
    }
}
add_action('wp_enqueue_scripts', 'little_hands_checkout_styles');

// AJAX handler for removing cart items on checkout
function remove_cart_item_ajax() {
    // Check nonce security
    check_ajax_referer('wc_store_api', 'nonce');

    $cart_item_key = sanitize_text_field($_POST['cart_item_key']);

    if (WC()->cart->remove_cart_item($cart_item_key)) {
        // Calculate new cart info
        WC()->cart->calculate_totals();

        wp_send_json_success(array(
            'message' => 'Item removed from cart',
            'cart_count' => WC()->cart->get_cart_contents_count(),
            'cart_total' => WC()->cart->get_cart_total(),
            'fragments' => apply_filters('woocommerce_add_to_cart_fragments', array())
        ));
    } else {
        wp_send_json_error(array('message' => 'Failed to remove item from cart'));
    }
}
add_action('wp_ajax_remove_cart_item', 'remove_cart_item_ajax');
add_action('wp_ajax_nopriv_remove_cart_item', 'remove_cart_item_ajax');

// Debug and add remove button to checkout
function debug_checkout_cart_item_name($product_name, $cart_item, $cart_item_key) {
    // Always log this to see if the filter is being called
    error_log('Cart item name filter called: ' . $product_name . ' | Checkout: ' . (is_checkout() ? 'YES' : 'NO'));

    if (is_checkout()) {
        $remove_button = ' <a href="' . esc_url(wc_get_cart_remove_url($cart_item_key)) . '"
                            class="remove-from-checkout"
                            data-cart_item_key="' . esc_attr($cart_item_key) . '"
                            data-product_id="' . esc_attr($cart_item['product_id']) . '"
                            title="Remove this item"
                            style="color: #dc3545; text-decoration: none; font-size: 18px; font-weight: bold;">[×]</a>';
        return $product_name . $remove_button;
    }
    return $product_name;
}
add_filter('woocommerce_cart_item_name', 'debug_checkout_cart_item_name', 10, 3);

// Alternative: Add to checkout review order directly with real cart keys
function add_checkout_remove_buttons() {
    if (is_checkout()) {
        $cart_items = array();
        $index = 0;
        foreach (WC()->cart->get_cart() as $cart_item_key => $cart_item) {
            $cart_items[] = array(
                'key' => $cart_item_key,
                'index' => $index++
            );
        }
        ?>
        <script type="text/javascript">
        var cartItemKeys = <?php echo json_encode($cart_items); ?>;

        jQuery(document).ready(function($) {
            console.log('Adding remove buttons to checkout items...', cartItemKeys);

            function addButtons() {
                // WooCommerce Block-based checkout structure
                var $blockItems = $('.wc-block-components-order-summary-item');
                console.log('Found block items:', $blockItems.length);

                // Classic checkout fallback
                var $table = $('.woocommerce-checkout-review-order-table, .shop_table, table.shop_table');
                var $rows = $('.cart_item, tr.cart_item');
                console.log('Found classic tables:', $table.length, 'rows:', $rows.length);

                if ($blockItems.length > 0) {
                    // Handle WooCommerce blocks
                    $blockItems.each(function(index) {
                        var $item = $(this);
                        var $productName = $item.find('.wc-block-components-product-name');
                        var productNameText = $productName.text().trim();
                        console.log('Block item ' + index + ':', productNameText, 'length:', productNameText.length);

                        // Only add button if we have actual content and not already added
                        if ($productName.length && productNameText.length > 0 && !$productName.find('.remove-from-checkout').length && cartItemKeys[index]) {
                            var cartItemData = cartItemKeys[index];
                            var cartItemKey = cartItemData.key;

                            var removeButton = '<a href="#" class="remove-from-checkout manual-remove block-remove" ' +
                                             'data-cart_item_key="' + cartItemKey + '" ' +
                                             'data-row-index="' + index + '" ' +
                                             'title="Remove this item" ' +
                                             'style="' +
                                                'display: inline-flex; ' +
                                                'align-items: center; ' +
                                                'justify-content: center; ' +
                                                'width: 20px; ' +
                                                'height: 20px; ' +
                                                'background: #f8f9fa; ' +
                                                'color: #6c757d; ' +
                                                'border: 1px solid #e9ecef; ' +
                                                'border-radius: 50%; ' +
                                                'text-decoration: none; ' +
                                                'font-size: 12px; ' +
                                                'font-weight: 400; ' +
                                                'line-height: 1; ' +
                                                'margin-left: 8px; ' +
                                                'transition: all 0.2s ease; ' +
                                                'opacity: 0.7; ' +
                                             '">×</a>';

                            $productName.append(removeButton);
                            console.log('Added block remove button to item ' + index + ' (' + productNameText + ') with key: ' + cartItemKey);
                        }
                    });
                } else if ($rows.length > 0) {
                    // Handle classic checkout
                    $rows.each(function(index) {
                        var $row = $(this);
                        var $nameCell = $row.find('.product-name, td:first-child, td.product-name');

                        if ($nameCell.length && !$nameCell.find('.remove-from-checkout').length && cartItemKeys[index]) {
                            var cartItemData = cartItemKeys[index];
                            var cartItemKey = cartItemData.key;

                            var removeButton = '<a href="#" class="remove-from-checkout manual-remove classic-remove" ' +
                                             'data-cart_item_key="' + cartItemKey + '" ' +
                                             'data-row-index="' + index + '" ' +
                                             'title="Remove this item" ' +
                                             'style="color: #dc3545; text-decoration: none; margin-left: 10px; font-size: 18px; font-weight: bold;">[×]</a>';

                            $nameCell.append(removeButton);
                            console.log('Added classic remove button to item ' + index + ' with key: ' + cartItemKey);
                        }
                    });
                }

                setTimeout(function() {
                    console.log('Manual buttons found:', $('.manual-remove').length);
                    console.log('Block buttons:', $('.block-remove').length, 'Classic buttons:', $('.classic-remove').length);
                }, 100);
            }

            // Use MutationObserver to detect when content loads
            function waitForContent() {
                var observer = new MutationObserver(function(mutations) {
                    var contentLoaded = false;
                    mutations.forEach(function(mutation) {
                        if (mutation.type === 'childList' || mutation.type === 'characterData') {
                            // Check if any product names now have content
                            $('.wc-block-components-product-name').each(function() {
                                if ($(this).text().trim().length > 0) {
                                    contentLoaded = true;
                                }
                            });
                        }
                    });

                    if (contentLoaded) {
                        console.log('Content detected, adding buttons...');
                        addButtons();
                    }
                });

                // Observe the checkout sidebar for changes
                var checkoutSidebar = document.querySelector('.wc-block-checkout__sidebar');
                if (checkoutSidebar) {
                    observer.observe(checkoutSidebar, {
                        childList: true,
                        subtree: true,
                        characterData: true
                    });
                }

                // Also set a backup timer
                setTimeout(function() {
                    observer.disconnect();
                    addButtons();
                }, 5000);
            }

            // Try immediately and also set up content watching
            addButtons();
            setTimeout(addButtons, 500);
            setTimeout(addButtons, 1000);
            setTimeout(addButtons, 2000);
            setTimeout(addButtons, 3000);

            // Start mutation observer
            waitForContent();
        });
        </script>
        <?php
    }
}
add_action('wp_footer', 'add_checkout_remove_buttons');

// E-Transfer Settings for Admin
function little_hands_etransfer_settings() {
    add_options_page(
        'E-Transfer Settings',
        'E-Transfer Settings',
        'manage_options',
        'etransfer-settings',
        'little_hands_etransfer_settings_page'
    );
}
add_action('admin_menu', 'little_hands_etransfer_settings');

function little_hands_etransfer_settings_page() {
    ?>
    <div class="wrap">
        <h1>E-Transfer Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('etransfer_settings');
            do_settings_sections('etransfer-settings');
            ?>
            <table class="form-table">
                <tr>
                    <th scope="row">Recipient Email</th>
                    <td>
                        <input type="email" name="etransfer_email" value="<?php echo esc_attr(get_option('etransfer_email', 'hello@littlehandsstudio.com')); ?>" class="regular-text" />
                        <p class="description">Email address to receive e-Transfers</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Security Question</th>
                    <td>
                        <input type="text" name="etransfer_question" value="<?php echo esc_attr(get_option('etransfer_question', 'What is this for?')); ?>" class="regular-text" />
                        <p class="description">Security question for e-Transfer (if required)</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Security Answer</th>
                    <td>
                        <input type="text" name="etransfer_answer" value="<?php echo esc_attr(get_option('etransfer_answer', 'artbox')); ?>" class="regular-text" />
                        <p class="description">Answer to the security question</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Payment Deadline</th>
                    <td>
                        <input type="number" name="etransfer_deadline" value="<?php echo esc_attr(get_option('etransfer_deadline', '24')); ?>" class="small-text" min="1" max="168" /> hours
                        <p class="description">Time limit for payment (1-168 hours)</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Additional Notes</th>
                    <td>
                        <textarea name="etransfer_notes" rows="4" cols="50" class="large-text"><?php echo esc_textarea(get_option('etransfer_notes', '(You can customize or remove this if auto-deposit is enabled)')); ?></textarea>
                        <p class="description">Additional instructions or notes for customers</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Enable E-Transfer Instructions</th>
                    <td>
                        <label for="etransfer_enabled">
                            <input type="checkbox" name="etransfer_enabled" id="etransfer_enabled" value="1" <?php checked(1, get_option('etransfer_enabled', 1)); ?> />
                            Show e-Transfer instructions on order confirmation page
                        </label>
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

// Register settings
function little_hands_register_etransfer_settings() {
    register_setting('etransfer_settings', 'etransfer_email', 'sanitize_email');
    register_setting('etransfer_settings', 'etransfer_question', 'sanitize_text_field');
    register_setting('etransfer_settings', 'etransfer_answer', 'sanitize_text_field');
    register_setting('etransfer_settings', 'etransfer_deadline', 'absint');
    register_setting('etransfer_settings', 'etransfer_notes', 'sanitize_textarea_field');
    register_setting('etransfer_settings', 'etransfer_enabled', 'absint');
}
add_action('admin_init', 'little_hands_register_etransfer_settings');

// Add sibling upsell to checkout
function add_checkout_sibling_upsell() {
    if (!is_checkout() || is_wc_endpoint_url()) {
        return;
    }

    // Check if sibling product is already in cart
    $sibling_product_id = 169; // Sibling product ID
    $has_sibling = false;

    foreach (WC()->cart->get_cart() as $cart_item) {
        if ($cart_item['product_id'] == $sibling_product_id) {
            $has_sibling = true;
            break;
        }
    }

    // Only show upsell if sibling not in cart and main products are in cart
    if (!$has_sibling && WC()->cart->get_cart_contents_count() > 0) {
        $sibling_product = wc_get_product($sibling_product_id);
        if ($sibling_product) {
            ?>
            <div class="checkout-sibling-upsell">
                <h3>🎨 Add a Sibling Kit?</h3>
                <div class="sibling-upsell-content">
                    <div class="sibling-info">
                        <h4><?php echo $sibling_product->get_name(); ?></h4>
                        <p class="sibling-description"><?php echo $sibling_product->get_description() ?: $sibling_product->get_short_description(); ?></p>
                        <div class="sibling-price">
                            <span class="price-label">Only</span>
                            <span class="price-amount"><?php echo $sibling_product->get_price_html(); ?></span>
                        </div>
                    </div>
                    <div class="sibling-actions">
                        <button type="button" class="btn btn-primary add-sibling-checkout" data-product-id="<?php echo $sibling_product_id; ?>">
                            <i class="fas fa-plus"></i>
                            Add to Order
                        </button>
                        <button type="button" class="btn btn-link dismiss-sibling-upsell">
                            No Thanks
                        </button>
                    </div>
                </div>
            </div>
            <?php
        }
    }
}
add_action('woocommerce_checkout_before_order_review', 'add_checkout_sibling_upsell');

// AJAX handler for adding sibling product during checkout
function add_sibling_to_cart_checkout() {
    check_ajax_referer('wc_store_api', 'nonce');

    $product_id = intval($_POST['product_id']);
    $added = WC()->cart->add_to_cart($product_id, 1);

    if ($added) {
        wp_send_json_success(array(
            'message' => 'Sibling kit added to cart!',
            'cart_count' => WC()->cart->get_cart_contents_count(),
            'fragments' => apply_filters('woocommerce_add_to_cart_fragments', array())
        ));
    } else {
        wp_send_json_error(array('message' => 'Failed to add sibling kit'));
    }
}
add_action('wp_ajax_add_sibling_checkout', 'add_sibling_to_cart_checkout');
add_action('wp_ajax_nopriv_add_sibling_checkout', 'add_sibling_to_cart_checkout');

// AJAX handler for fetching category products dynamically
function get_lhs_category_products() {
    // Debug logging
    error_log('LHS AJAX called for category: ' . (isset($_POST['category']) ? $_POST['category'] : 'none'));

    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'lhs_products_nonce')) {
        error_log('LHS AJAX: Nonce verification failed');
        wp_send_json_error(array('message' => 'Security check failed'));
    }

    // Check if WooCommerce is active
    if (!class_exists('WooCommerce')) {
        error_log('LHS AJAX: WooCommerce not active');
        wp_send_json_error(array('message' => 'WooCommerce not available'));
    }

    $category = sanitize_text_field($_POST['category']);
    $html = '';

    if ($category === 'packages') {
        // Get Project Boxes (variable product ID 164)
        $product = wc_get_product(164);
        error_log('LHS AJAX: Package product found: ' . ($product ? 'yes' : 'no'));

        if ($product && $product->is_type('variable')) {
            $variations = $product->get_available_variations();
            error_log('LHS AJAX: Found ' . count($variations) . ' variations');

            foreach ($variations as $variation_data) {
                $variation = wc_get_product($variation_data['variation_id']);
                $variation_name = implode(' ', $variation_data['attributes']);

                $html .= '<div class="lhs-product-option" data-product-id="164" data-variation-id="' . $variation_data['variation_id'] . '" data-price="' . $variation->get_price() . '">';
                $html .= '<div class="lhs-product-option-content">';
                $html .= '<h3>' . esc_html($variation_name) . '</h3>';
                $html .= '<div class="price">$' . number_format($variation->get_price(), 2) . '</div>';
                $html .= '<p>' . esc_html($variation->get_description() ?: 'No description available') . '</p>';
                $html .= '</div>';
                $html .= '<button class="lhs-add-to-cart">Add to Cart</button>';
                $html .= '</div>';
            }
        } else {
            error_log('LHS AJAX: Product 164 not found or not variable');
        }
    } elseif ($category === 'subscriptions') {
        // Get Subscriptions (product ID 190)
        $product = wc_get_product(190);
        error_log('LHS AJAX: Subscription product found: ' . ($product ? 'yes' : 'no'));
        error_log('LHS AJAX: Subscription product type: ' . ($product ? $product->get_type() : 'none'));

        if ($product) {
            if ($product->is_type('variable')) {
                // Handle variable subscription product
                $variations = $product->get_available_variations();
                error_log('LHS AJAX: Found ' . count($variations) . ' subscription variations');

                foreach ($variations as $variation_data) {
                    $variation = wc_get_product($variation_data['variation_id']);
                    $variation_name = implode(' ', $variation_data['attributes']);

                    $html .= '<div class="lhs-product-option" data-product-id="190" data-variation-id="' . $variation_data['variation_id'] . '" data-price="' . $variation->get_price() . '">';
                    $html .= '<div class="lhs-product-option-content">';
                    $html .= '<h3>' . esc_html($variation_name) . '</h3>';
                    $html .= '<div class="price">$' . number_format($variation->get_price(), 2) . '/month</div>';
                    $html .= '<p>' . esc_html($variation->get_description() ?: 'Automatic monthly art box deliveries') . '</p>';
                    $html .= '</div>';
                    $html .= '<button class="lhs-add-to-cart">Subscribe</button>';
                    $html .= '</div>';
                }
            } else {
                // Handle simple subscription product
                $html .= '<div class="lhs-product-option" data-product-id="190" data-price="' . $product->get_price() . '">';
                $html .= '<div class="lhs-product-option-content">';
                $html .= '<h3>' . esc_html($product->get_name()) . '</h3>';
                $html .= '<div class="price">$' . number_format($product->get_price(), 2) . '/month</div>';
                $html .= '<p>' . esc_html($product->get_short_description() ?: 'Automatic monthly art box deliveries') . '</p>';
                $html .= '</div>';
                $html .= '<button class="lhs-add-to-cart">Subscribe</button>';
                $html .= '</div>';
            }
        }
    } elseif ($category === 'events') {
        // Get Events (product ID 195)
        $product = wc_get_product(195);
        error_log('LHS AJAX: Event product found: ' . ($product ? 'yes' : 'no'));
        error_log('LHS AJAX: Event product type: ' . ($product ? $product->get_type() : 'none'));

        if ($product) {
            if ($product->is_type('variable')) {
                // Handle variable event product
                $variations = $product->get_available_variations();
                error_log('LHS AJAX: Found ' . count($variations) . ' event variations');

                foreach ($variations as $variation_data) {
                    $variation = wc_get_product($variation_data['variation_id']);
                    $variation_name = implode(' ', $variation_data['attributes']);

                    $html .= '<div class="lhs-product-option" data-product-id="195" data-variation-id="' . $variation_data['variation_id'] . '" data-price="' . $variation->get_price() . '">';
                    $html .= '<div class="lhs-product-option-content">';
                    $html .= '<h3>' . esc_html($variation_name) . '</h3>';
                    $html .= '<div class="price">$' . number_format($variation->get_price(), 2) . '</div>';
                    $html .= '<p>' . esc_html($variation->get_description() ?: 'Special occasion art experiences') . '</p>';
                    $html .= '</div>';
                    $html .= '<button class="lhs-add-to-cart">Book Event</button>';
                    $html .= '</div>';
                }
            } else {
                // Handle simple event product
                $html .= '<div class="lhs-product-option" data-product-id="195" data-price="' . $product->get_price() . '">';
                $html .= '<div class="lhs-product-option-content">';
                $html .= '<h3>' . esc_html($product->get_name()) . '</h3>';
                $html .= '<div class="price">$' . number_format($product->get_price(), 2) . '</div>';
                $html .= '<p>' . esc_html($product->get_short_description() ?: 'Special occasion art experiences') . '</p>';
                $html .= '</div>';
                $html .= '<button class="lhs-add-to-cart">Book Event</button>';
                $html .= '</div>';
            }
        }
    }

    error_log('LHS AJAX: Generated HTML length: ' . strlen($html));

    if ($html) {
        wp_send_json_success(array('html' => $html));
    } else {
        error_log('LHS AJAX: No HTML generated, returning error');
        wp_send_json_error(array('message' => 'No products found for category: ' . $category));
    }
}
add_action('wp_ajax_get_lhs_category_products', 'get_lhs_category_products');
add_action('wp_ajax_nopriv_get_lhs_category_products', 'get_lhs_category_products');

// AJAX handler for adding product to cart with sibling addon
function lhs_add_to_cart_with_sibling() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wc_store_api')) {
        wp_send_json_error(array('message' => 'Security check failed'));
    }

    // Check if WooCommerce is active
    if (!class_exists('WooCommerce')) {
        wp_send_json_error(array('message' => 'WooCommerce not active'));
    }

    $product_id = intval($_POST['product_id']);
    $variation_id = !empty($_POST['variation_id']) ? intval($_POST['variation_id']) : null;
    $sibling_product_id = 169; // Sibling addon product ID

    try {
        // Clear cart first to ensure clean checkout
        WC()->cart->empty_cart();

        // Add main product
        if ($variation_id) {
            $cart_item_key = WC()->cart->add_to_cart($product_id, 1, $variation_id);
        } else {
            $cart_item_key = WC()->cart->add_to_cart($product_id, 1);
        }

        if (!$cart_item_key) {
            wp_send_json_error(array('message' => 'Failed to add main product to cart'));
        }

        // Add sibling addon
        $sibling_cart_key = WC()->cart->add_to_cart($sibling_product_id, 1);

        if (!$sibling_cart_key) {
            wp_send_json_error(array('message' => 'Failed to add sibling addon to cart'));
        }

        // Calculate totals
        WC()->cart->calculate_totals();

        wp_send_json_success(array(
            'message' => 'Products added to cart successfully',
            'checkout_url' => wc_get_checkout_url(),
            'cart_count' => WC()->cart->get_cart_contents_count()
        ));

    } catch (Exception $e) {
        error_log('LHS Add to Cart Error: ' . $e->getMessage());
        wp_send_json_error(array('message' => 'Failed to add products to cart'));
    }
}
add_action('wp_ajax_lhs_add_to_cart_with_sibling', 'lhs_add_to_cart_with_sibling');
add_action('wp_ajax_nopriv_lhs_add_to_cart_with_sibling', 'lhs_add_to_cart_with_sibling');

// AJAX handler for adding single product to cart
function lhs_add_to_cart_single() {
    // Debug logging
    error_log('LHS Single Cart AJAX called with data: ' . print_r($_POST, true));

    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wc_store_api')) {
        error_log('LHS Single Cart: Nonce verification failed');
        wp_send_json_error(array('message' => 'Security check failed'));
    }

    // Check if WooCommerce is active
    if (!class_exists('WooCommerce')) {
        error_log('LHS Single Cart: WooCommerce not active');
        wp_send_json_error(array('message' => 'WooCommerce not active'));
    }

    $product_id = intval($_POST['product_id']);
    $variation_id = !empty($_POST['variation_id']) ? intval($_POST['variation_id']) : null;

    error_log('LHS Single Cart: Product ID: ' . $product_id . ', Variation ID: ' . ($variation_id ?: 'none'));

    try {
        // Clear cart first to ensure clean checkout
        WC()->cart->empty_cart();

        // Add main product
        if ($variation_id) {
            $cart_item_key = WC()->cart->add_to_cart($product_id, 1, $variation_id);
        } else {
            $cart_item_key = WC()->cart->add_to_cart($product_id, 1);
        }

        if (!$cart_item_key) {
            wp_send_json_error(array('message' => 'Failed to add product to cart'));
        }

        // Calculate totals
        WC()->cart->calculate_totals();

        wp_send_json_success(array(
            'message' => 'Product added to cart successfully',
            'checkout_url' => wc_get_checkout_url(),
            'cart_count' => WC()->cart->get_cart_contents_count()
        ));

    } catch (Exception $e) {
        error_log('LHS Add to Cart Error: ' . $e->getMessage());
        wp_send_json_error(array('message' => 'Failed to add product to cart'));
    }
}
add_action('wp_ajax_lhs_add_to_cart_single', 'lhs_add_to_cart_single');
add_action('wp_ajax_nopriv_lhs_add_to_cart_single', 'lhs_add_to_cart_single');

// Add modal HTML to all pages
function lhs_add_global_modals() {
    ?>
    <!-- Sibling Addon Popup -->
    <div id="sibling-popup" class="sibling-popup" style="display: none;">
        <div class="sibling-popup-content">
            <h3>Add a Sibling Kit?</h3>
            <p>Perfect for siblings to create alongside! Includes additional supplies for one more child.</p>
            <div class="sibling-popup-price">
                <strong>+$8.95</strong>
            </div>
            <div class="sibling-popup-buttons">
                <button id="add-sibling" class="btn-primary">Yes, Add Sibling Kit</button>
                <button id="skip-sibling" class="btn-secondary">No Thanks</button>
            </div>
        </div>
        <div class="sibling-popup-overlay"></div>
    </div>

    <!-- Simple Modal -->
    <div id="lhs-modal" class="lhs-modal">
        <div class="lhs-modal-overlay"></div>
        <div class="lhs-modal-content">
            <button class="lhs-modal-close">&times;</button>
            <div id="modal-step-1" class="modal-step">
                <h2>Choose Your Creative Experience</h2>
                <p>What type of art adventure are you looking for?</p>
                <div class="lhs-category-options">
                    <div class="lhs-category-option" data-category="packages">
                        <h3>Packages</h3>
                        <p>One-time project boxes</p>
                    </div>
                    <div class="lhs-category-option" data-category="subscriptions">
                        <h3>Subscriptions</h3>
                        <p>Monthly recurring deliveries</p>
                    </div>
                    <div class="lhs-category-option" data-category="events">
                        <h3>Events</h3>
                        <p>Special occasion experiences</p>
                    </div>
                </div>
            </div>

            <div id="modal-step-2" class="modal-step" style="display: none;">
                <h2>Select Your Option</h2>
                <div id="lhs-product-options"></div>
                <button class="lhs-back-btn">← Back</button>
            </div>
        </div>
    </div>
    <?php
}
add_action('wp_footer', 'lhs_add_global_modals');