const teamInput = document.getElementById('teamInput');
const favoritesIcon = document.getElementById('favoritesIcon');
const resultsDiv = document.getElementById('results');
const favoritesContainer = document.getElementById('favoritesContainer');
let currentSortingOption = 'az'; // Default sorting option

const sortButton = document.getElementById('sortButton');
const sortIcon = sortButton.querySelector('i');

sortButton.addEventListener('click', function() {
  // Cycle through sorting options
  switch(currentSortingOption) {
    case 'az':
      currentSortingOption = 'za';
      sortIcon.className = 'fas fa-sort-alpha-down-alt'; // Z-A sort icon
      sortButton.title = 'Sort Z-A'; // hover text
      break;
    case 'za':
      currentSortingOption = 'league';
      sortIcon.className = 'fas fa-globe'; // Sort by league icon
      sortButton.title = 'Sort by League'; // hover text
      break;
    case 'league':
      currentSortingOption = 'az';
      sortIcon.className = 'fas fa-sort-alpha-down'; // A-Z sort icon
      sortButton.title = 'Sort A-Z'; // hover text
      break;
    default:
      currentSortingOption = 'az';
      sortIcon.className = 'fas fa-sort-alpha-down'; // A-Z sort icon
      sortButton.title = 'Sort A-Z'; // hover text
  }

  showFavorites(currentSortingOption);
});


function sortFavorites(sortOption) {
  if (sortOption === 'az') {
    favorites.sort((a, b) => a.name.localeCompare(b.name));
  } else if (sortOption === 'za') {
    favorites.sort((a, b) => b.name.localeCompare(a.name));
  } else if (sortOption === 'league') {
    favorites.sort((a, b) => a.league.localeCompare(b.league));
  }

  showFavorites();
}

function showFavorites() {
  const favoritesList = document.getElementById('favoritesList');
  favoritesList.innerHTML = ''; // Clear previous favorites

  if (favorites.length === 0) {
    favoritesList.textContent = 'No favorites selected.';
    return;
  }

  let sortedFavorites = [...favorites]; // Create a copy of favorites array

  switch (currentSortingOption) {
    case 'az':
      sortedFavorites.sort((a, b) => a.name.localeCompare(b.name)); // Sort A-Z
      break;
    case 'za':
      sortedFavorites.sort((a, b) => b.name.localeCompare(a.name)); // Sort Z-A
      break;
    case 'league':
      sortedFavorites.sort((a, b) => a.league.localeCompare(b.league)); // Sort by League
      break;
    default:
      sortedFavorites.sort((a, b) => a.name.localeCompare(b.name)); // Default sorting option: A-Z
  }

  sortedFavorites.forEach(team => {
    let teamCard = createTeamCard(team);
    favoritesList.appendChild(teamCard);
  });
}


let favoritesMode = false;
let favorites = [];

function getLeagueAbbreviation(leagueName) {
  // replace apostrophes and lower case the string for comparison
  let standardizedLeagueName = leagueName.replace(/’/g, "'").toLowerCase();
  switch (standardizedLeagueName) {
    case 'national hockey league':
      return 'NHL';
    case 'major league baseball':
      return 'MLB';
    case 'national basketball association':
      return 'NBA';
    case 'national football league':
      return 'NFL';
    case 'major league soccer':
      return 'MLS';
    case 'canadian football league':
      return 'CFL';
    case 'western hockey league':
      return 'WHL';
    case 'american hockey league':
      return 'AHL';
    case 'national lacrosse league':
      return 'NLL';
    case 'national women\'s soccer league':
      return 'NWSL';
    case 'women\'s national basketball association':
      return 'WNBA';
    case 'professional golfers\' association':
      return 'PGA';
    case 'major league rugby':
      return 'MLR';
    case 'quebec major junior hockey league':
      return 'QMJHL';
    // add more cases here as needed
    default:
      return leagueName;
  }
}




// Function to clear search query and results
function clearSearch() {
  teamInput.value = ''; // Clear search query
  resultsDiv.innerHTML = ''; // Clear previous results
}

function saveFavorites() {
  localStorage.setItem('favorites', JSON.stringify(favorites));
}

function exportFavorites(format) {
  let exportContent = '';
  if (format === 'json') {
    exportContent = JSON.stringify(favorites.map(({ _id, name, league, colors }) => ({
      name,
      league,
      colors: colors.map(({ _id: colorId, name: colorName, hex }) => ({ hex }))
    })));
    downloadFile(exportContent, 'favorites.json', 'application/json');
  } else if (format === 'csv') {
    exportContent = convertFavoritesToCSV();
    downloadFile(exportContent, 'favorites.csv', 'text/csv');
  }
}

function downloadFile(content, fileName, contentType) {
  const element = document.createElement('a');
  const file = new Blob([content], { type: contentType });
  element.href = URL.createObjectURL(file);
  element.download = fileName;
  element.click();
}

function convertFavoritesToCSV() {
  let csvContent = 'Team Name, League, Hex Colors\n';
  favorites.forEach((favorite) => {
    const { name, league, colors } = favorite;
    const teamName = name.replace(/,/g, ''); // Remove commas from the team name
    const leagueName = league.replace(/,/g, ''); // Remove commas from the league name
    const hexColors = colors.map(color => color.hex).join(','); // Combine hex colors with commas
    csvContent += `${teamName},${leagueName},${hexColors}\n`;
  });
  return csvContent;
}

const exportJSONButton = document.getElementById('exportJSONButton');
const exportCSVButton = document.getElementById('exportCSVButton');

exportJSONButton.addEventListener('click', function () {
  exportFavorites('json');
});

exportCSVButton.addEventListener('click', function () {
  exportFavorites('csv');
});


function loadFavorites() {
  let storedFavorites = localStorage.getItem('favorites');
  if (storedFavorites) {
    favorites = JSON.parse(storedFavorites);
  }
}

// Event listener for search input changes
teamInput.addEventListener('input', function () {
  const query = teamInput.value;
  if (query === '') {
    // Clear search query and results
    clearSearch();
    if (favoritesMode) {
      // Show favorites when favorites mode is enabled and search query is empty
      showFavorites();
    }
    return;
  }

  if (query.length >= 3) {
    if (favoritesMode) {
      // Filter favorites based on the search term
      let filteredFavorites = favorites.filter(
        (team) =>
          team.name.toLowerCase().includes(query.toLowerCase()) ||
          team.league.toLowerCase().includes(query.toLowerCase())
      );
      resultsDiv.innerHTML = ''; // Clear previous results
      if (filteredFavorites.length === 0) {
        resultsDiv.innerHTML = 'No favorites match your search.';
      } else {
        filteredFavorites.forEach((team) => {
          let teamCard = createTeamCard(team);
          resultsDiv.appendChild(teamCard);
        });
      }
    } else {
      // Search for teams by name
      fetch(`https://quilled-pointed-diagram.glitch.me/teams?name=${encodeURIComponent(query)}`)
        .then((response) => response.json())
        .then((data) => {
          resultsDiv.innerHTML = ''; // Clear previous results
          if (data.length === 0) {
            resultsDiv.innerHTML = 'No results found.';
          } else {
            data.forEach((team) => {
              let teamCard = createTeamCard(team);
              resultsDiv.appendChild(teamCard);
            });
          }
        })
        .catch((error) => {
          console.error('Error:', error);
        });
    }
  } else {
    resultsDiv.innerHTML = ''; // Clear previous results
  }
});
// Event listener for league pills
// Event listener for league pills
const leaguePillsContainer = document.querySelector('.league-pills');

// Event listener for league pills
leaguePillsContainer.addEventListener('click', function (event) {
  if (event.target.classList.contains('league-pill')) {
    const leaguePills = leaguePillsContainer.querySelectorAll('.league-pill');
    const selectedLeaguePill = event.target;

    // Toggle active state of league pills
    leaguePills.forEach((pill) => {
      if (pill === selectedLeaguePill) {
        pill.classList.toggle('active');
      } else {
        pill.classList.remove('active');
      }
    });

    const league = selectedLeaguePill.dataset.league;

    if (selectedLeaguePill.classList.contains('active') && favoritesMode) {
      // Show favorite teams from the selected league
      const filteredFavorites = favorites.filter((team) => team.league === league);
      resultsDiv.innerHTML = ''; // Clear previous results
      if (filteredFavorites.length === 0) {
        resultsDiv.innerHTML = 'No favorites match your selection.';
      } else {
        filteredFavorites.forEach((team) => {
          let teamCard = createTeamCard(team);
          resultsDiv.appendChild(teamCard);
        });
      }
    } else if (selectedLeaguePill.classList.contains('active')) {
      // Show all teams from the selected league
      fetch(`https://quilled-pointed-diagram.glitch.me/teams/league/${encodeURIComponent(league)}`)
        .then((response) => response.json())
        .then((data) => {
          resultsDiv.innerHTML = ''; // Clear previous results
          if (data.length === 0) {
            resultsDiv.innerHTML = 'No results found.';
          } else {
            data.forEach((team) => {
              let teamCard = createTeamCard(team);
              resultsDiv.appendChild(teamCard);
            });
          }
        })
        .catch((error) => {
          console.error('Error:', error);
        });
    } else if (favoritesMode) {
      // Show favorites from all leagues
      showFavorites();
    } else {
      // Hide teams when no league pill is selected
      resultsDiv.innerHTML = '';
    }
  }
});


// Event listener for "Show All" button
const showAllButton = document.createElement('span');
showAllButton.textContent = 'Show All';
showAllButton.classList.add('league-pill');
let showAllMode = false; // Track the state of the "Show All" button

showAllButton.addEventListener('click', function () {
  const leaguePills = leaguePillsContainer.querySelectorAll('.league-pill');

  // Remove active state from all league pills
  leaguePills.forEach((pill) => {
    pill.classList.remove('active');
  });

  if (!showAllMode) {
    // Show all teams
    fetch('https://quilled-pointed-diagram.glitch.me/teams')
      .then((response) => response.json())
      .then((data) => {
        resultsDiv.innerHTML = ''; // Clear previous results
        if (data.length === 0) {
          resultsDiv.innerHTML = 'No results found.';
        } else {
          data.sort((a, b) => a.name.localeCompare(b.name)); // Sort teams alphabetically by name
          data.forEach((team) => {
            let teamCard = createTeamCard(team);
            resultsDiv.appendChild(teamCard);
          });
        }
      })
      .catch((error) => {
        console.error('Error:', error);
      });
    showAllMode = true;
  } else {
    // Hide all teams
    resultsDiv.innerHTML = '';
    showAllMode = false;
  }
});

leaguePillsContainer.appendChild(showAllButton);

// Event listener for favorites button click
favoritesIcon.addEventListener('click', function () {
  favoritesMode = !favoritesMode;
  if (favoritesMode) {
    favoritesContainer.style.display = 'block';
    favoritesIcon.classList.add('active');
    clearSearch(); // Clear search query when entering favorites mode
    showFavorites();
  } else {
    favoritesContainer.style.display = 'none';
    favoritesIcon.classList.remove('active');
    clearSearch(); // Clear search query when leaving favorites mode
  }
});


// Function to create a team card
function createTeamCard(team) {
  let teamCard = document.createElement('div');
  teamCard.classList.add('card', 'my-3');

 
  let teamHeaderDiv = document.createElement('div');
  let gradient = `linear-gradient(to right, ${team.colors[0].hex} 70%, #000000)`;
  teamHeaderDiv.style.background = gradient;
  teamHeaderDiv.style.height = '30px';
  teamHeaderDiv.style.color = '#fff'; // Setting the text color to white for better contrast
  teamHeaderDiv.classList.add('d-flex', 'namediv', 'justify-content-between', 'p-2', 'team-header');

  let teamNameDiv = document.createElement('div');
  teamNameDiv.classList.add('team-name-div');

  let teamNameText = document.createElement('span');
  teamNameText.textContent = team.name;
  teamNameText.classList.add('team-name');

  let leagueDiv = document.createElement('div');
  leagueDiv.classList.add('league-div');

  let leagueText = document.createElement('span');
  leagueText.textContent = getLeagueAbbreviation(team.league);
  leagueText.classList.add('league-name');

  teamNameDiv.appendChild(teamNameText);
  leagueDiv.appendChild(leagueText);

  teamHeaderDiv.appendChild(teamNameDiv);
  teamHeaderDiv.appendChild(leagueDiv);
  teamCard.setAttribute('data-team', team.name);
  teamCard.appendChild(teamHeaderDiv);

  let cardBody = document.createElement('div');
  cardBody.classList.add('card-body', 'd-flex', 'flex-column');

  let colorDivs = document.createElement('div');
  colorDivs.classList.add('d-flex', 'justify-content-start');

  team.colors.forEach(color => {
    let colorDiv = document.createElement('div');
    colorDiv.style.background = color.hex;
    colorDiv.style.width = '30px';
    colorDiv.style.height = '30px';
    colorDiv.style.marginRight = '10px';
    colorDiv.classList.add('color');
    colorDiv.setAttribute('data-toggle', 'tooltip');
    colorDiv.setAttribute('data-original-title', color.hex);

    colorDiv.addEventListener('click', function () {
      navigator.clipboard.writeText(color.hex);
      showCopiedMessage(colorDiv);
    });

    // Initialize the tooltip for the new colorDiv
    $(colorDiv).tooltip();

    colorDivs.appendChild(colorDiv);
  });

  let plusMinusDiv = document.createElement('div');
  plusMinusDiv.classList.add('plus-minus-div', 'mt-auto');

  let plusIcon = document.createElement('i');
  plusIcon.classList.add('fas', 'fa-plus', 'plus-icon');
  plusIcon.addEventListener('click', function () {
    toggleFavorite(team);
    plusIcon.classList.toggle('fa-plus');
    plusIcon.classList.toggle('fa-minus');
    if (favoritesMode) {
      if (isFavorite(team)) {
        let teamCard = createTeamCard(team);
        resultsDiv.appendChild(teamCard);
      } else {
        let teamCard = resultsDiv.querySelector(`[data-team="${team.name}"]`);
        if (teamCard) {
          resultsDiv.removeChild(teamCard);
        }
      }
      showFavorites();
    }
  });

  if (isFavorite(team)) {
    teamCard.classList.add('favorited');
    plusIcon.classList.replace('fa-plus', 'fa-minus');
  }

  plusMinusDiv.appendChild(plusIcon);
  cardBody.appendChild(colorDivs);
  cardBody.appendChild(plusMinusDiv);
  teamCard.appendChild(cardBody);

  return teamCard;
}


function toggleFavorite(team) {
  const index = favorites.findIndex(favorite => favorite.name === team.name);
  if (index === -1) {
    favorites.push(team);
  } else {
    favorites.splice(index, 1);
  }
  saveFavorites();

  // Check if favorites array is empty
  if (favorites.length === 0) {
    favoritesIcon.classList.remove('active');
  }
}


// Load favorites from local storage at the beginning
loadFavorites();

// Function to check if a team is favorited
function isFavorite(team) {
  return favorites.some(favorite => favorite.name === team.name);
}

function showFavorites() {
  resultsDiv.innerHTML = ''; // Clear previous results

  if (favorites.length === 0) {
    resultsDiv.textContent = 'No favorites selected.';
    return;
  }

  let sortedFavorites = favorites.slice(); // Create a copy of favorites array

  // Sort the favorites based on the selected sorting option
  if (currentSortingOption === 'az') {
    sortedFavorites.sort((a, b) => a.name.localeCompare(b.name));
  } else if (currentSortingOption === 'za') {
    sortedFavorites.sort((a, b) => b.name.localeCompare(a.name));
  } else if (currentSortingOption === 'league') {
    sortedFavorites.sort((a, b) => a.league.localeCompare(b.league) || a.name.localeCompare(b.name));
  }

  sortedFavorites.forEach(team => {
    let teamCard = createTeamCard(team);
    resultsDiv.appendChild(teamCard);
  });
}

// Function to show a "Copied to clipboard" message for a color
function showCopiedMessage(colorDiv) {
  const originalTitle = colorDiv.getAttribute('data-original-title');
  colorDiv.setAttribute('data-original-title', 'Copied to clipboard');
  $(colorDiv).tooltip('hide').tooltip('show');

  setTimeout(function () {
    colorDiv.setAttribute('data-original-title', originalTitle);
    $(colorDiv).tooltip('hide');
  }, 2000);
}

$(document).ready(function () {
  $(document).on('mouseenter', '.color', function () {
    $(this).css('opacity', '0.7');
  });

  $(document).on('mouseleave', '.color', function () {
    $(this).css('opacity', '1');
  });
});

// Load theme from local storage
loadTheme();

document.getElementById('theme-toggle').addEventListener('click', function () {
  const body = document.body;
  if (body.classList.contains('theme-light')) {
    body.classList.replace('theme-light', 'theme-dark');
    localStorage.setItem('theme', 'theme-dark');
  } else {
    body.classList.replace('theme-dark', 'theme-light');
    localStorage.setItem('theme', 'theme-light');
  }
});

function loadTheme() {
  const theme = localStorage.getItem('theme') || 'theme-light';
  document.body.classList.add(theme);
}



